// --- DATA VARIABLES ---
let netwatchData = [];
let zoom;
let svg,
    g;
let selectedNodeId = null;
let knownDownDevices = new Set(); // Track devices that have already triggered a toast

// --- AUTO REFRESH CONFIG ---
let refreshInterval = null;
const REFRESH_RATE = 10000; // 10 Seconds

// --- ELEMENTS ---
const themeBtn = document.getElementById('theme-btn');
const html = document.documentElement;
const navPanel = document.getElementById('nav-panel');
const deviceMenu = document.getElementById('device-menu');
const connectSection = document.getElementById('connect-section');
const statsSection = document.getElementById('stats-section');
const lineStylePanel = document.getElementById('line-style-panel');
const connectForm = document.getElementById('connect-form');
const disconnectBtn = document.getElementById('disconnect-btn');
const manualRefreshBtn = document.getElementById('manual-refresh-btn');
const graphContainer = document.getElementById('graph-container');
const toastContainer = document.getElementById('toast-container');
const addDeviceBtn = document.getElementById('add-device-btn');

// --- THEME & PERSISTENCE ---
function applyTheme(theme) {
    if (theme === 'dark') {
        html
            .classList
            .add('dark');
    } else {
        html
            .classList
            .remove('dark');
    }
}

const savedTheme = localStorage.getItem('theme') || (
    window.matchMedia('(prefers-color-scheme: dark)').matches
        ? 'dark'
        : 'light'
);
applyTheme(savedTheme);

themeBtn.addEventListener('click', () => {
    const currentTheme = html
        .classList
        .contains('dark')
            ? 'dark'
            : 'light';
    const newTheme = currentTheme === 'dark'
        ? 'light'
        : 'dark';
    applyTheme(newTheme);
    localStorage.setItem('theme', newTheme);
    if (svg) 
        renderGraph();
    }
);

// --- TOAST NOTIFICATIONS ---
function showToast(message, type = 'info') {
    const toast = document.createElement('div');

    let bgClass = 'bg-gray-800';
    let icon = 'fa-info-circle';
    let textClass = 'text-white';

    if (type === 'error') {
        bgClass = 'bg-red-600';
        icon = 'fa-exclamation-circle';
    } else if (type === 'success') {
        bgClass = 'bg-green-600';
        icon = 'fa-check-circle';
    } else if (type === 'warning') {
        bgClass = 'bg-yellow-600';
        icon = 'fa-exclamation-triangle';
    }

    toast.className = `toast ${bgClass} ${textClass} p-3 rounded shadow-lg flex items-center gap-3 min-w-[300px] border border-white/10`;
    toast.innerHTML = `
        <i class="fas ${icon} text-lg"></i>
        <div class="text-sm font-bold">${message}</div>
    `;

    toastContainer.appendChild(toast);

    // Remove after 4 seconds
    setTimeout(() => {
        toast.style.opacity = '0';
        toast.style.transform = 'translateX(100%)';
        toast.style.transition = 'all 0.3s';
        setTimeout(() => toast.remove(), 300);
    }, 4000);
}

// --- MENUS & PANELS ---
function toggleNav(show, e) {
    if (e) 
        e.stopPropagation();
    if (show) {
        navPanel
            .classList
            .add('active');
    } else {
        navPanel
            .classList
            .remove('active');
    }
}
window.toggleNav = toggleNav;

// Global Context Menu (Right Click)
window.addEventListener('contextmenu', (e) => {
    // 1. If clicking a node, let D3 handle it (don't show sidebar)
    if (e.target.closest('.node')) {
        return;
    }

    // 2. If clicking inside the menus themselves, do nothing
    if (e.target.closest('#device-menu') || e.target.closest('#nav-panel')) {
        return;
    }

    // 3. Prevent Default Browser Menu
    e.preventDefault();

    // 4. Hide the device specific menu if it's open
    toggleMenu(deviceMenu, false);

    // 5. Show the Sidebar (No positionMenu calculation needed anymore)
    toggleNav(true);
});

// Update the toggleNav function to handle the new class logic
function toggleNav(show, e) {
    if (e) 
        e.stopPropagation();
    
    if (show) {
        navPanel
            .classList
            .add('active'); // CSS handles the slide-in
    } else {
        navPanel
            .classList
            .remove('active'); // CSS handles the slide-out
    }
}

// Ensure it's global
window.toggleNav = toggleNav;

// Keep the global click listener to close sidebar when clicking canvas
window.addEventListener('click', (e) => {
    // If we click outside nav-panel, close it
    if (!navPanel.contains(e.target) && navPanel.classList.contains('active')) {
        toggleNav(false);
    }

    // Existing logic for device menu
    if (!deviceMenu.contains(e.target)) 
        toggleMenu(deviceMenu, false);
    }
);

function positionMenu(menu, x, y) {
    const rect = menu.getBoundingClientRect();
    const finalX = (x + menu.offsetWidth > window.innerWidth)
        ? x - menu.offsetWidth
        : x;
    const finalY = (y + menu.offsetHeight > window.innerHeight)
        ? y - menu.offsetHeight
        : y;
    menu.style.left = `${finalX}px`;
    menu.style.top = `${finalY}px`;
}

function toggleMenu(menu, show) {
    if (show) {
        menu
            .classList
            .add('active');
    } else {
        menu
            .classList
            .remove('active');
    }
}

window.addEventListener('click', (e) => {
    if (!navPanel.contains(e.target)) 
        toggleMenu(navPanel, false);
    if (!deviceMenu.contains(e.target)) 
        toggleMenu(deviceMenu, false);
    }
);

// --- CONNECT / DISCONNECT LOGIC ---

connectForm.addEventListener('submit', (e) => {
    e.preventDefault();

    const formData = new FormData();
    formData.append('host', document.getElementById('mk-addr').value);
    formData.append('port', document.getElementById('mk-port').value);
    formData.append('user', document.getElementById('mk-user').value);
    formData.append('pass', document.getElementById('mk-pass').value);

    // Send POST to connect.php
    fetch('connect.php', {
        method: 'POST',
        body: formData
    })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') {
                // Save persistence flag
                localStorage.setItem('mk_connected', 'true');
                showToast("Connected to Mikrotik successfully", "success");

                // UI Updates
                connectSection
                    .classList
                    .add('hidden');
                statsSection
                    .classList
                    .remove('hidden');
                if (lineStylePanel) 
                    lineStylePanel
                        .classList
                        .remove('hidden');
                
                toggleMenu(navPanel, false);

                // Start Monitoring
                fetchNetwatchData(true);

                if (refreshInterval) 
                    clearInterval(refreshInterval);
                refreshInterval = setInterval(() => fetchNetwatchData(true), REFRESH_RATE);
            } else {
                localStorage.removeItem('mk_connected');
                window.location.href = 'disconnect.php';
            }
        })
        .catch(error => {
            console.error("Connection Error:", error);
            localStorage.removeItem('mk_connected');
            window.location.href = 'disconnect.php';
        });
});

manualRefreshBtn.addEventListener('click', () => {
    fetchNetwatchData(true);
});

disconnectBtn.addEventListener('click', () => {
    localStorage.removeItem('mk_connected'); // Clear persistence
    window.location.href = 'disconnect.php';
});

// --- DATA FETCHING ---
async function fetchNetwatchData(redirectOnError = true) {
    try {
        const response = await fetch('netwatch.php', {method: 'POST'});

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const contentType = response
            .headers
            .get("content-type");
        if (!contentType || !contentType.includes("application/json")) {
            throw new Error("Received non-JSON response from server");
        }

        const data = await response.json();

        // Re-affirm connection on success
        localStorage.setItem('mk_connected', 'true');

        loadData(data);

        // --- SUCCESS STATE UI ---
        connectSection
            .classList
            .add('hidden');
        statsSection
            .classList
            .remove('hidden');
        if (lineStylePanel) 
            lineStylePanel
                .classList
                .remove('hidden');

        }
    catch (e) {
        console.error("Netwatch API Fetch Error:", e);

        // CHECK IF WE SHOULD PREVENT REDIRECT
        const isPreviouslyConnected = localStorage.getItem('mk_connected') === 'true';

        if (isPreviouslyConnected) {
            // Show toast instead of redirecting
            showToast("Connection to Mikrotik API failed! Retrying...", "error");
            // We return here to keep the current canvas/UI state active and not trigger the
            // disconnect logic below.
            return;
        }

        // Standard disconnect logic for initial load or manual disconnect
        if (redirectOnError) {
            window.location.href = 'disconnect.php';
        } else {
            // --- DISCONNECTED STATE UI (Initial Load) ---

            // 1. Show the Connect Form
            connectSection
                .classList
                .remove('hidden');

            // 2. Hide other connected UI elements
            statsSection
                .classList
                .add('hidden');
            if (lineStylePanel) 
                lineStylePanel
                    .classList
                    .add('hidden');
            if (refreshInterval) 
                clearInterval(refreshInterval);
            
            // 3. Clear the graph container and add instruction text
            if (graphContainer) {
                renderBlankState();
            }
            if (svg) {
                svg.remove();
                svg = null;
            }
        }
    }
}

// --- GRAPH LOGIC ---

function loadData(data) {
    const processed = data.map(d => {
        let parentId = "";
        try {
            if (d.comment) {
                const c = JSON.parse(d.comment);
                parentId = c.parent || "";
            }
        } catch (e) {
            parentId = "";
        }

        return {
            ...d,
            id: d['.id'],
            parentId: parentId
        };
    });

    // Ensure hierarchy integrity
    const idSet = new Set(processed.map(d => d.id));
    processed.forEach(d => {
        if (d.parentId !== "" && !idSet.has(d.parentId)) {
            d.parentId = "";
        }
        if (d.parentId === "") 
            d.parentId = null;
        }
    );

    netwatchData = processed;
    updateStats();
    renderGraph();
}

function updateStats() {
    const total = netwatchData.length;
    const activeNodes = netwatchData.filter(d => d.disabled === 'false');
    const up = activeNodes
        .filter(d => d.status === 'up')
        .length;
    const downNodes = activeNodes.filter(d => d.status === 'down');
    const down = downNodes.length;

    document
        .getElementById('stat-total')
        .innerText = total;
    document
        .getElementById('stat-up')
        .innerText = up;
    document
        .getElementById('stat-down')
        .innerText = down;

    // --- DOWN DEVICE TOAST LOGIC ---
    const currentDownIds = new Set();

    downNodes.forEach(node => {
        currentDownIds.add(node.id);

        // If this node is down, and we haven't warned about it yet
        if (!knownDownDevices.has(node.id)) {
            showToast(`Device "${node.name || node.host}" is DOWN!`, 'warning');
        }
    });

    // Update the global set.
    knownDownDevices = currentDownIds;
}

function renderBlankState() {
    graphContainer.innerHTML = `
        <div class="absolute inset-0 flex flex-col items-center justify-center pointer-events-none select-none">
            <div class="text-xl opacity-50 font-bold">No Devices Found</div>
            <div class="text-sm opacity-40 mt-2 flex items-center gap-2">
                <i class="fas fa-mouse-pointer"></i> Right Click to Open Menu / Connect
            </div>
        </div>
    `;
}

// --- RENDER (Update Pattern) ---
function renderGraph() {
    // 1. Check Data (Only runs if a successful connection was established)
    if (netwatchData.length === 0) {
        if (!svg) 
            renderBlankState();
        return;
    }

    if (!svg) 
        graphContainer.innerHTML = '';
    
    const width = graphContainer.clientWidth;
    const height = graphContainer.clientHeight;
    const lineStyle = document
        .getElementById('line-style-select')
        .value;

    // 2. Prepare Hierarchy
    const stratify = d3
        .stratify()
        .id(d => d.id)
        .parentId(d => d.parentId);

    let root;
    try {
        root = stratify(netwatchData);
    } catch (e) {
        const masterRoot = {
            id: "root",
            name: "Network",
            status: "up",
            parentId: null
        };
        const patchedData = [
            masterRoot, ...netwatchData.map(d => ({
                ...d,
                parentId: d.parentId || "root"
            }))
        ];
        root = d3
            .stratify()
            .id(d => d.id)
            .parentId(d => d.parentId)(patchedData);
    }

    // 3. Layout
    const treeLayout = d3
        .tree()
        .nodeSize([120, 100]);
    treeLayout(root);

    // 4. Initialize SVG (ONCE)
    if (!svg) {
        zoom = d3
            .zoom()
            .scaleExtent([0.1, 4])
            .on("zoom", (e) => g.attr("transform", e.transform));

        svg = d3
            .select("#graph-container")
            .append("svg")
            .attr("width", width)
            .attr("height", height)
            .call(zoom)
            .on("dblclick.zoom", null);

        g = svg.append("g");

        // Center Graph
        const initialTransform = d3
            .zoomIdentity
            .translate(width / 2, 50)
            .scale(0.8);
        svg.call(zoom.transform, initialTransform);
    }

    // 5. Link Generator
    let linkGen;
    if (lineStyle === 'straight') {
        linkGen = d => `M${d.source.x},${d.source.y} L${d.target.x},${d.target.y}`;
    } else if (lineStyle === 'step') {
        linkGen = d => `M${d.source.x},${d.source.y} V${ (d.source.y + d.target.y) / 2} H${d.target.x} V${d.target.y}`;
    } else {
        linkGen = d3
            .linkVertical()
            .x(d => d.x)
            .y(d => d.y);
    }

    // 6. Draw Links
    const links = g
        .selectAll(".link")
        .data(root.links(), d => d.target.data.id);

    links.join(
        enter => enter.append("path")
        // Apply status class (up/down/unknown)
            .attr("class", d => `link ${d.target.data.status}`).attr("d", linkGen).attr(
                "stroke-opacity",
                d => (d.target.data.disabled === 'true')
                    ? 0.2
                    : 1
            ),

        update => update.attr("class", d => `link ${d.target.data.status}`).transition().duration(
            1000
        ).attr("d", linkGen).attr(
            "stroke-opacity",
            d => (d.target.data.disabled === 'true')
                ? 0.2
                : 1
        ),

        exit => exit.remove()
    );

    // 7. Draw Nodes
    const nodes = g
        .selectAll(".node")
        .data(root.descendants().filter(d => d.data.id !== 'root'), d => d.data.id);

    nodes.join(enter => {
        const nodeEnter = enter
            .append("g")
            .attr(
                "class",
                d => `node ${d.data.status} ${d.data.disabled === 'true'
                    ? 'disabled'
                    : ''}`
            )
            .attr("transform", d => `translate(${d.x},${d.y})`)
            .on("mouseenter", showTooltip)
            .on("mouseleave", hideTooltip)
            .on("click", (e, d) => openEditModal(e, d))
            .on("contextmenu", (e, d) => {
                e.preventDefault();
                e.stopPropagation();
                hideTooltip();

                toggleMenu(navPanel, false);
                if (!d || !d.data || !d.data.id) 
                    return;
                selectedNodeId = d.data.id;

                if (selectedNodeId === 'root') 
                    return;
                
                positionMenu(deviceMenu, e.clientX, e.clientY);
                toggleMenu(deviceMenu, true);
            });

        nodeEnter
            .append("circle")
            .attr("r", 15)
            .attr("stroke", "#fff")
            // COLOR LOGIC: Green for Up, Red for Down, Grey for anything else
            .attr("fill", d => {
                if (d.data.status === 'up') 
                    return '#22c55e';
                if (d.data.status === 'down') 
                    return '#ef4444';
                return '#9ca3af';
            });

        nodeEnter
            .append("text")
            .attr("dy", 30)
            .attr("text-anchor", "middle")
            .text(d => d.data.name);

        return nodeEnter;
    }, update => {
        update
            .attr(
                "class",
                d => `node ${d.data.status} ${d.data.disabled === 'true'
                    ? 'disabled'
                    : ''}`
            )
            .transition()
            .duration(1000)
            .attr("transform", d => `translate(${d.x},${d.y})`);

        update
            .select("circle")
            // COLOR LOGIC (Update phase)
            .attr("fill", d => {
                if (d.data.status === 'up') 
                    return '#22c55e';
                if (d.data.status === 'down') 
                    return '#ef4444';
                return '#9ca3af';
            });

        update
            .select("text")
            .text(d => d.data.name);

        return update;
    }, exit => exit.remove());
}

// --- DEVICE ACTIONS ---
function deviceAction(action) {
    if (!selectedNodeId) 
        return;
    toggleMenu(deviceMenu, false);

    if (action === 'addChild') {
        // Pass the selected node ID as the parent for the new device
        openEditModal(null, null, selectedNodeId);
        return;
    }

    const apiUrl = `netwatch.php?action=${action}&id=${selectedNodeId}`;

    fetch(apiUrl, {method: 'POST'})
        .then(response => {
            if (!response.ok) 
                throw new Error(`HTTP error! status: ${response.status}`);
            return response.json();
        })
        .then(() => fetchNetwatchData(true))
        .catch(e => {
            console.error(`Action (${action}) Error:`, e);
            showToast(`Action failed: ${e.message}`, "error");
        });
}

// --- LINE STYLE PERSISTENCE ---
const lineStyleSelect = document.getElementById('line-style-select');
const savedLineStyle = localStorage.getItem('lineStyle') || 'curved';
lineStyleSelect.value = savedLineStyle;

lineStyleSelect.addEventListener('change', () => {
    const newLineStyle = lineStyleSelect.value;
    localStorage.setItem('lineStyle', newLineStyle);
    if (svg) 
        renderGraph();
    }
);

// --- ZOOM & UI HELPERS ---
window.zoomIn = () => svg && svg
    .transition()
    .call(zoom.scaleBy, 1.2);
window.zoomOut = () => svg && svg
    .transition()
    .call(zoom.scaleBy, 0.8);
window.resetZoom = () => {
    if (!svg) 
        return;
    const initialTransform = d3
        .zoomIdentity
        .translate(graphContainer.clientWidth / 2, 50)
        .scale(0.8);
    svg
        .transition()
        .duration(750)
        .call(zoom.transform, initialTransform);
}

// Tooltip
const tooltip = document.getElementById('tooltip');
function showTooltip(event, d) {
    if (!d || !d.data || !d.data.id) 
        return;
    
    const data = netwatchData.find(n => n.id === d.data.id) || d.data;

    let statusColor = 'text-gray-400';
    if (data.status === 'up') 
        statusColor = 'text-green-400';
    if (data.status === 'down') 
        statusColor = 'text-red-400';
    
    tooltip.innerHTML = `
        <div class="font-bold border-b border-gray-600 pb-1 mb-1">${data.name}</div>
        <div>Host: ${data.host}</div>
        <div>Status: <span class="${statusColor} uppercase font-bold">${data.status} ${data.disabled === 'true'
        ? '(Disabled)'
        : ''}</span></div>
        <div>Last Change: ${data.since}</div>
    `;
    tooltip.style.left = (event.pageX + 15) + "px";
    tooltip.style.top = (event.pageY + 15) + "px";
    tooltip.style.opacity = 1;
}

function hideTooltip() {
    tooltip.style.opacity = 0;
}

// --- EDIT/CREATE MODAL ---
const editModal = document.getElementById('edit-modal');

// 1. "Add Device" Button Listener
if (addDeviceBtn) {
    addDeviceBtn.addEventListener('click', () => {
        // Open in create mode, no specific parent
        openEditModal(null, null, null);
    });
}

function openEditModal(event, d, parentIdForNewNode = null) {
    toggleMenu(deviceMenu, false);

    const isEdit = !!d && !!d.data;

    // Get Data (if editing)
    const data = isEdit
        ? netwatchData.find(n => n.id === d.data.id)
        : null;

    // UI Text Updates
    document
        .getElementById('edit-modal-title')
        .textContent = isEdit
            ? 'Edit Device'
            : 'Add New Device';
    document
        .getElementById('edit-submit-btn')
        .textContent = isEdit
            ? 'Update Device'
            : 'Create Device';
    document
        .getElementById('edit-form')
        .dataset
        .mode = isEdit
            ? 'edit'
            : 'create';

    // Populate Fields
    document
        .getElementById('edit-id')
        .value = isEdit
            ? data.id
            : '';
    document
        .getElementById('edit-name')
        .value = isEdit
            ? data.name
            : '';
    document
        .getElementById('edit-host')
        .value = isEdit
            ? data.host
            : '';
    // Defaults for new devices
    document
        .getElementById('edit-interval')
        .value = isEdit
            ? data.interval
            : '90';
    document
        .getElementById('edit-timeout')
        .value = isEdit
            ? data.timeout
            : '45';
    document
        .getElementById('edit-up')
        .value = isEdit
            ? data['up-script']
            : '';
    document
        .getElementById('edit-down')
        .value = isEdit
            ? data['down-script']
            : '';

    // Populate Parent Dropdown
    const parentSelect = document.getElementById('edit-parent');
    parentSelect.innerHTML = '<option value="">(Root)</option>';

    // Logic to select the correct parent
    let selectedParentId = '';
    if (isEdit) {
        selectedParentId = data.parentId || '';
    } else if (parentIdForNewNode) {
        selectedParentId = parentIdForNewNode; // From Right-click "Add Child"
    }

    netwatchData.forEach(node => {
        // Prevent setting itself as parent (if editing)
        if (isEdit && node.id === data.id) 
            return;
        
        // Exclude root virtual node from list
        if (node.id === 'root') 
            return;
        
        const sel = (node.id === selectedParentId)
            ? 'selected'
            : '';
        parentSelect.innerHTML += `<option value="${node.id}" ${sel}>${node.name}</option>`;
    });

    editModal
        .classList
        .remove('hidden');
}

function closeEditModal() {
    editModal
        .classList
        .add('hidden');
}
window.closeEditModal = closeEditModal;

// 2. Unified Form Submission
document.getElementById('edit-form').addEventListener('submit', (e) => {
    e.preventDefault();
    const mode = document.getElementById('edit-form').dataset.mode;

    // --- CREATE MODE ---
    if (mode === 'create') {
        const formData = new FormData();
        formData.append('name', document.getElementById('edit-name').value);
        formData.append('host', document.getElementById('edit-host').value);
        formData.append('parent', document.getElementById('edit-parent').value);
        formData.append('up-script', document.getElementById('edit-up').value);
        formData.append('down-script', document.getElementById('edit-down').value);

        fetch('netwatch-add.php', {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            if (data.status === 'success') {
                closeEditModal();
                fetchNetwatchData(true);
                showToast("Device Created Successfully", "success");
            } else {
                throw new Error(data.message || "Unknown error");
            }
        })
        .catch(e => showToast(e.message, "error"));
    
    // --- EDIT MODE (Updated Logic) ---
    } else {
        const formData = new FormData();
        
        // Append fields matching netwatch-update.php expectations
        formData.append('id', document.getElementById('edit-id').value);
        formData.append('name', document.getElementById('edit-name').value);
        formData.append('host', document.getElementById('edit-host').value);
        formData.append('parent', document.getElementById('edit-parent').value); // Use 'parent', not 'parentid'
        formData.append('up-script', document.getElementById('edit-up').value);
        formData.append('down-script', document.getElementById('edit-down').value);

        fetch('netwatch-update.php', {
            method: 'POST',
            body: formData
        })
        .then(response => {
             // Check if network response is OK
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            // Check RouterOS specific success status
            if (data.status === 'success') {
                closeEditModal();
                fetchNetwatchData(true); // Refresh canvas
                showToast("Device Updated Successfully", "success");
            } else {
                // Handle RouterOS trap errors
                throw new Error(data.message || "Failed to update device");
            }
        })
        .catch(e => {
            console.error(e);
            showToast(e.message || "Error updating device", "error");
        });
    }
});

// --- INITIALIZATION ---
fetchNetwatchData(false).then(() => {
    // Only start refresh if the initial check was successful (i.e., we are
    // connected)
    if (connectSection.classList.contains('hidden')) {
        if (refreshInterval) 
            clearInterval(refreshInterval);
        refreshInterval = setInterval(() => fetchNetwatchData(true), REFRESH_RATE);
    }
});