const API_URL = 'api.php'; 
let data = { packages: [], clients: [], billings: [], users: [] };
let currentUser = null; 

// --- 1. Session Management ---
window.addEventListener('DOMContentLoaded', () => {
    checkSession();
});

function checkSession() {
    const session = localStorage.getItem('sumberejo_session');
    if (session) {
        currentUser = JSON.parse(session);
        loadData().then((success) => {
            if(success) initAppUI();
            else logout();
        });
    } else {
        showLogin();
    }
}

function saveSession(user) {
    currentUser = user;
    localStorage.setItem('sumberejo_session', JSON.stringify(currentUser));
}

function logout() {
    localStorage.removeItem('sumberejo_session');
    currentUser = null;
    showLogin();
    showToast('Signed out successfully', 'info');
}

function showLogin() {
    document.getElementById('app-view').classList.add('d-none-custom');
    document.getElementById('login-view').classList.remove('d-none-custom');
}

function initAppUI() {
    document.getElementById('user-display-name').textContent = currentUser.name;
    document.getElementById('user-display-role').textContent = currentUser.role === 'admin' ? 'Administrator' : 'Client';
    document.getElementById('user-avatar').textContent = currentUser.name.charAt(0).toUpperCase();

    document.getElementById('login-view').classList.add('d-none-custom');
    document.getElementById('app-view').classList.remove('d-none-custom');
    
    if (currentUser.role === 'client') {
        // Hide Admin Menus
        document.getElementById('menu-admin').classList.add('d-none-custom');
        document.getElementById('menu-client').classList.remove('d-none-custom');
        
        // Hide Admin Pages classes
        document.querySelectorAll('.admin-page').forEach(el => el.classList.add('d-none-custom'));
        
        // Show Client Dashboard
        renderClientDashboard();
        showPage('client-dashboard');
    } else {
        // Admin View
        document.getElementById('menu-admin').classList.remove('d-none-custom');
        document.getElementById('menu-client').classList.add('d-none-custom');
        refreshUI(); // Standard Admin Render
    }
}

// --- 2. Client Dashboard Logic ---
function renderClientDashboard() {
    const myClient = data.clients[0]; // API returns array with only 1 client for 'client' role
    if (!myClient) {
        showToast("Profile not found", "error");
        return;
    }

    // 1. Fill Profile Form
    document.getElementById('cpName').value = myClient.name;
    document.getElementById('cpAddress').value = myClient.address;
    document.getElementById('cpLat').value = myClient.lat || '';
    document.getElementById('cpLng').value = myClient.lng || '';
    
    // 2. Fill Package Info
    const myPkg = data.packages.find(p => p.id === myClient.packageId);
    if (myPkg) {
        document.getElementById('cpPackageName').textContent = myPkg.name;
        document.getElementById('cpPackageSpeed').textContent = myPkg.speed;
        document.getElementById('cpPackagePrice').textContent = parseInt(myPkg.price).toLocaleString();
    } else {
        document.getElementById('cpPackageName').textContent = "Unknown";
    }
    document.getElementById('cpStatus').textContent = myClient.status;

    // 3. Fill Billing History
    const tbody = document.querySelector('#clientBillingTable tbody');
    tbody.innerHTML = '';
    
    // Sort bills by date desc
    const myBills = data.billings.sort((a,b) => new Date(b.dueDate) - new Date(a.dueDate));
    
    if(myBills.length === 0) {
        tbody.innerHTML = `<tr><td colspan="4" class="p-4 text-center text-gray-400">No billing history found.</td></tr>`;
    } else {
        myBills.forEach(bill => {
            const statusClass = bill.status === 'Paid' ? 'text-green-600 bg-green-50' : 'text-yellow-600 bg-yellow-50';
            const statusIcon = bill.status === 'Paid' ? '<i class="fas fa-check-circle mr-1"></i>' : '<i class="fas fa-clock mr-1"></i>';
            
            tbody.innerHTML += `
                <tr class="hover:bg-gray-50/50 transition">
                    <td class="p-4 font-medium text-gray-800">${bill.type} ${bill.month !== '-' ? '('+bill.month+')' : ''}</td>
                    <td class="p-4">${bill.dueDate}</td>
                    <td class="p-4 font-bold">Rp ${parseInt(bill.amount).toLocaleString()}</td>
                    <td class="p-4 text-right">
                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${statusClass}">
                            ${statusIcon} ${bill.status}
                        </span>
                    </td>
                </tr>
            `;
        });
    }
}

// Client Profile Save Handler
document.getElementById('clientProfileForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    const btn = e.target.querySelector('button');
    setBtnLoading(btn, true);

    // Update local data object
    if (data.clients[0]) {
        data.clients[0].name = document.getElementById('cpName').value;
        data.clients[0].address = document.getElementById('cpAddress').value;
        data.clients[0].lat = document.getElementById('cpLat').value;
        data.clients[0].lng = document.getElementById('cpLng').value;
    }

    const success = await saveData();
    setBtnLoading(btn, false);
    if (success) showToast('Profile updated successfully');
});


// --- 3. UX Helpers ---
function showToast(message, type = 'success') {
    const container = document.getElementById('toast-container');
    let bgClass, icon;
    if (type === 'success') {
        bgClass = 'bg-white border-l-4 border-green-500 text-gray-800';
        icon = '<i class="fas fa-check-circle text-green-500 text-xl"></i>';
    } else if (type === 'error') {
        bgClass = 'bg-white border-l-4 border-red-500 text-gray-800';
        icon = '<i class="fas fa-exclamation-circle text-red-500 text-xl"></i>';
    } else {
        bgClass = 'bg-white border-l-4 border-blue-500 text-gray-800';
        icon = '<i class="fas fa-info-circle text-blue-500 text-xl"></i>';
    }

    const toast = document.createElement('div');
    toast.className = `${bgClass} shadow-lg rounded-r-lg p-4 flex items-center gap-3 transform transition-all duration-300 translate-x-full opacity-0 min-w-[300px]`;
    toast.innerHTML = `<div>${icon}</div><div class="font-medium text-sm">${message}</div>`;
    container.appendChild(toast);
    requestAnimationFrame(() => toast.classList.remove('translate-x-full', 'opacity-0'));
    setTimeout(() => {
        toast.classList.add('translate-x-full', 'opacity-0');
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

function setBtnLoading(btn, isLoading) {
    if (isLoading) {
        if (!btn.dataset.originalText) btn.dataset.originalText = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-circle-notch fa-spin mr-2"></i> Processing...';
        btn.disabled = true;
        btn.classList.add('opacity-75', 'cursor-not-allowed');
    } else {
        btn.innerHTML = btn.dataset.originalText;
        btn.disabled = false;
        btn.classList.remove('opacity-75', 'cursor-not-allowed');
    }
}

// --- 4. Login Logic ---
document.getElementById('loginForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    const btn = e.target.querySelector('button');
    const email = document.getElementById('loginEmail').value;
    const pass = document.getElementById('loginPassword').value;

    setBtnLoading(btn, true);
    
    try {
        const response = await fetch(`${API_URL}?action=login`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ email: email, password: pass })
        });

        const resData = await response.json();

        if (response.ok && resData.status === 'success') {
            saveSession(resData.user);
            await loadData();
            initAppUI();
            showToast(`Welcome back, ${resData.user.name}!`);
            setTimeout(() => setBtnLoading(btn, false), 500); 
        } else {
            throw new Error(resData.message || 'Login failed');
        }
    } catch (error) {
        showToast(error.message, 'error');
        setBtnLoading(btn, false);
    }
});

// --- 5. Data Logic ---
async function loadData() {
    if (!currentUser) return false;
    try {
        const response = await fetch(`${API_URL}?action=get_data&user_id=${currentUser.id}`);
        if (!response.ok) throw new Error('Failed to load data');
        const loadedData = await response.json();
        
        data.packages = loadedData.packages || [];
        data.clients = loadedData.clients || [];
        data.billings = loadedData.billings || [];
        data.users = loadedData.users || [];
        return true;
    } catch (error) {
        console.error("Error loading data:", error);
        showToast("Failed to load data from server", "error");
        return false;
    }
}

async function saveData() {
    if(!currentUser) return false;
    try {
        await new Promise(r => setTimeout(r, 600));
        const response = await fetch(`${API_URL}?action=save_data&user_id=${currentUser.id}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });
        if(!response.ok) throw new Error("Save failed");
        
        // If admin, refresh standard UI. If client, stay on dashboard.
        if (currentUser.role === 'admin') refreshUI();
        return true;
    } catch (error) { 
        console.error("Error saving:", error); 
        showToast("Failed to save changes", "error");
        return false;
    }
}

// --- 6. Admin Render Logic (Existing) ---
function refreshUI() {
    if (currentUser.role !== 'admin') return; // Double check

    // Stats
    document.getElementById('stat-packages').textContent = data.packages.length;
    document.getElementById('stat-clients').textContent = data.clients.filter(c => c.status === 'Active').length;
    document.getElementById('stat-bills').textContent = data.billings.filter(b => b.status === 'Paid').length;

    renderPackages(true);
    renderClients(true);
    renderBillings(true);
    renderAccounts(); 
}

function renderPackages(isAdmin) {
    const tbody = document.querySelector('#packagesTable tbody');
    tbody.innerHTML = '';
    data.packages.forEach(pkg => {
        const actions = isAdmin ? 
            `<td class="p-4 text-right">
                <button class="text-blue-500 hover:text-blue-700 mx-1" onclick='openModal("packageModal", ${JSON.stringify(pkg)})'><i class="fas fa-pen"></i></button>
                <button class="text-red-500 hover:text-red-700 mx-1" onclick='promptDelete("packages", "${pkg.id}")'><i class="fas fa-trash"></i></button>
            </td>` : '';
        tbody.innerHTML += `<tr><td class="p-4 font-bold text-gray-800">${pkg.name}</td><td class="p-4"><span class="bg-gray-100 text-gray-800 text-xs px-2 py-1 rounded border border-gray-200">${pkg.speed} Mbps</span></td><td class="p-4 text-gray-500">Rp ${parseInt(pkg.price).toLocaleString()}</td>${actions}</tr>`;
    });
}

function renderClients(isAdmin) {
    const tbody = document.querySelector('#clientsTable tbody');
    tbody.innerHTML = '';
    data.clients.forEach(cli => {
        const pkg = data.packages.find(p => p.id == cli.packageId);
        const pkgName = pkg ? pkg.name : '<span class="text-red-500 text-xs">Deleted</span>';
        let badgeClass = cli.status === 'Active' ? 'bg-green-100 text-green-700' : 'bg-gray-200 text-gray-600';
        
        const actions = isAdmin ? 
            `<td class="p-4 text-right">
                <button class="text-yellow-500 hover:text-yellow-600 mx-1" title="Manage Login" onclick='openClientAccountModal(${JSON.stringify(cli)})'><i class="fas fa-user-lock"></i></button>
                <button class="text-blue-500 hover:text-blue-700 mx-1" title="Edit Details" onclick='openModal("clientModal", ${JSON.stringify(cli)})'><i class="fas fa-pen"></i></button>
                <button class="text-red-500 hover:text-red-700 mx-1" title="Delete" onclick='promptDelete("clients", "${cli.id}")'><i class="fas fa-trash"></i></button>
            </td>` : '';
        tbody.innerHTML += `<tr><td class="p-4 font-bold text-gray-800">${cli.name}</td><td class="p-4 text-gray-500 text-sm">${pkgName}</td><td class="p-4 text-gray-500 text-sm">${cli.address}</td><td class="p-4"><span class="${badgeClass} px-3 py-1 rounded-full text-xs font-medium">${cli.status}</span></td>${actions}</tr>`;
    });
}

function renderBillings(isAdmin) {
    const tbody = document.querySelector('#billingsTable tbody');
    tbody.innerHTML = '';
    data.billings.forEach(bill => {
        const cli = data.clients.find(c => c.id == bill.clientId);
        const cliName = cli ? cli.name : 'Unknown';
        const statusBadge = bill.status === 'Paid' ? 'bg-green-100 text-green-700' : 'bg-yellow-100 text-yellow-700';
        let details = `<span class="font-medium text-gray-800">${bill.type}</span>` + (bill.type === 'Monthly Bill' && bill.month ? ` <span class="text-gray-400 text-xs">(${bill.month})</span>` : '');
        const actions = isAdmin ? 
            `<td class="p-4 text-right"><button class="text-blue-500 hover:text-blue-700 mx-1" onclick='openModal("billingModal", ${JSON.stringify(bill)})'><i class="fas fa-pen"></i></button><button class="text-red-500 hover:text-red-700 mx-1" onclick='promptDelete("billings", "${bill.id}")'><i class="fas fa-trash"></i></button></td>` : '';
        tbody.innerHTML += `<tr><td class="p-4 font-bold text-gray-800">${cliName}</td><td class="p-4">${details}</td><td class="p-4 text-gray-500 font-medium">Rp ${parseInt(bill.amount).toLocaleString()}</td><td class="p-4 text-sm text-gray-500">${bill.dueDate}</td><td class="p-4"><span class="${statusBadge} px-3 py-1 rounded-full text-xs font-medium">${bill.status}</span></td>${actions}</tr>`;
    });
}

function renderAccounts() {
    const tbody = document.querySelector('#accountsTable tbody');
    tbody.innerHTML = '';
    data.users.forEach(user => {
        const roleBadge = user.role === 'admin' ? 'bg-blue-100 text-blue-700' : 'bg-gray-100 text-gray-700';
        const actions = `<td class="p-4 text-right">
            <button class="text-blue-500 hover:text-blue-700 mx-1" onclick='openModal("accountModal", ${JSON.stringify(user)})'><i class="fas fa-pen"></i></button>
            <button class="text-red-500 hover:text-red-700 mx-1" onclick='promptDelete("users", "${user.id}")'><i class="fas fa-trash"></i></button>
        </td>`;
        tbody.innerHTML += `
            <tr>
                <td class="p-4 font-bold text-gray-800">${user.name}</td>
                <td class="p-4 text-gray-500">${user.email}</td>
                <td class="p-4"><span class="${roleBadge} px-3 py-1 rounded-full text-xs font-medium uppercase">${user.role}</span></td>
                <td class="p-4 text-gray-400 text-sm">******</td>
                ${actions}
            </tr>`;
    });
}

// --- Helpers & Modals ---
function toggleSidebar() { 
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('sidebar-overlay');
    if (sidebar.classList.contains('-translate-x-full')) {
        sidebar.classList.remove('-translate-x-full');
        overlay.classList.remove('hidden');
    } else {
        sidebar.classList.add('-translate-x-full');
        overlay.classList.add('hidden');
    }
}

function showPage(pageName) {
    if (currentUser.role === 'client' && pageName !== 'client-dashboard') return;
    
    // Hide all pages
    document.querySelectorAll('[id^="page-"]').forEach(el => el.classList.add('d-none-custom'));
    
    // Deactivate all navs
    document.querySelectorAll('[id^="nav-"]').forEach(el => {
        el.classList.remove('bg-blue-50', 'text-blue-600'); 
        el.classList.add('text-gray-600');
    });

    // Show selected
    const targetPage = document.getElementById(`page-${pageName}`);
    if(targetPage) targetPage.classList.remove('d-none-custom');
    
    const activeNav = document.getElementById(`nav-${pageName}`);
    if(activeNav) {
        activeNav.classList.remove('text-gray-600');
        activeNav.classList.add('bg-blue-50', 'text-blue-600');
    }

    if(window.innerWidth < 1024) { 
        document.getElementById('sidebar').classList.add('-translate-x-full'); 
        document.getElementById('sidebar-overlay').classList.add('hidden'); 
    }
}

function generateId() { return Date.now().toString(36) + Math.random().toString(36).substr(2); }
function hideModal(id) { document.getElementById(id).classList.add('hidden'); }
function toggleMonthSelect() { 
    const type = document.getElementById('billType').value;
    const container = document.getElementById('billMonthContainer');
    if(type === 'Monthly Bill') container.classList.remove('hidden');
    else container.classList.add('hidden');
}

function populateSelect(elementId, items) { 
    const s = document.getElementById(elementId); 
    s.innerHTML = '<option value="">Select...</option>'; 
    items.forEach(i => s.innerHTML += `<option value="${i.id}">${i.name}</option>`); 
}

function openClientAccountModal(clientData) {
    if (currentUser.role !== 'admin') return;
    const existingUser = data.users.find(u => u.name === clientData.name);
    document.getElementById('caDisplayName').textContent = clientData.name;
    document.getElementById('caClientName').value = clientData.name;
    document.getElementById('caEmail').value = existingUser ? existingUser.email : '';
    document.getElementById('caPassword').value = existingUser ? existingUser.password : '';
    document.getElementById('clientAccountModal').classList.remove('hidden');
}

function openModal(id, editData = null) {
    if (currentUser.role !== 'admin') return; 
    if(id === 'packageModal') {
        document.getElementById('pkgId').value = editData ? editData.id : '';
        document.getElementById('pkgName').value = editData ? editData.name : '';
        document.getElementById('pkgSpeed').value = editData ? editData.speed : '';
        document.getElementById('pkgPrice').value = editData ? editData.price : '';
    } else if (id === 'clientModal') {
        populateSelect('cliPackage', data.packages);
        document.getElementById('cliId').value = editData ? editData.id : '';
        document.getElementById('cliName').value = editData ? editData.name : '';
        document.getElementById('cliPackage').value = editData ? editData.packageId : '';
        document.getElementById('cliAddress').value = editData ? editData.address : '';
        document.getElementById('cliLat').value = editData ? editData.lat : '';
        document.getElementById('cliLng').value = editData ? editData.lng : '';
        document.getElementById('cliStatus').value = editData ? editData.status : 'Active';
    } else if (id === 'billingModal') {
        populateSelect('billClient', data.clients);
        document.getElementById('billId').value = editData ? editData.id : '';
        document.getElementById('billClient').value = editData ? editData.clientId : '';
        document.getElementById('billType').value = editData ? editData.type : 'Monthly Bill';
        document.getElementById('billAmount').value = editData ? editData.amount : '';
        document.getElementById('billMonth').value = editData ? editData.month : 'January';
        document.getElementById('billDate').value = editData ? editData.dueDate : '';
        document.getElementById('billStatus').value = editData ? editData.status : 'Unpaid';
        toggleMonthSelect();
    } else if (id === 'accountModal') {
        document.getElementById('accId').value = editData ? editData.id : '';
        document.getElementById('accName').value = editData ? editData.name : '';
        document.getElementById('accEmail').value = editData ? editData.email : '';
        document.getElementById('accPassword').value = editData ? editData.password : '';
        document.getElementById('accRole').value = editData ? editData.role : 'client';
    }
    document.getElementById(id).classList.remove('hidden');
}

// --- Submit Handlers (Existing Admin Handlers) ---
document.getElementById('clientAccountForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const clientName = document.getElementById('caClientName').value;
    const email = document.getElementById('caEmail').value;
    const password = document.getElementById('caPassword').value;
    const userIndex = data.users.findIndex(u => u.name === clientName);
    if (userIndex >= 0) { data.users[userIndex].email = email; data.users[userIndex].password = password; } else { data.users.push({ id: generateId(), name: clientName, email: email, password: password, role: 'client' }); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('clientAccountModal'); showToast('Login credentials updated successfully'); }
});

document.getElementById('packageForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('pkgId').value;
    const newPkg = { id: id || generateId(), name: document.getElementById('pkgName').value, speed: document.getElementById('pkgSpeed').value, price: document.getElementById('pkgPrice').value };
    if (id) { data.packages[data.packages.findIndex(x => x.id == id)] = newPkg; } else { data.packages.push(newPkg); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('packageModal'); showToast('Package saved successfully'); }
});

document.getElementById('clientForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('cliId').value;
    const newCli = { id: id || generateId(), name: document.getElementById('cliName').value, packageId: document.getElementById('cliPackage').value, address: document.getElementById('cliAddress').value, lat: document.getElementById('cliLat').value, lng: document.getElementById('cliLng').value, status: document.getElementById('cliStatus').value };
    if (id) { data.clients[data.clients.findIndex(x => x.id == id)] = newCli; } else { data.clients.push(newCli); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('clientModal'); showToast('Client saved successfully'); }
});

document.getElementById('billingForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('billId').value;
    const type = document.getElementById('billType').value;
    const newBill = { id: id || generateId(), clientId: document.getElementById('billClient').value, type: type, amount: document.getElementById('billAmount').value, month: (type === 'Monthly Bill') ? document.getElementById('billMonth').value : '-', dueDate: document.getElementById('billDate').value, status: document.getElementById('billStatus').value };
    if (id) { data.billings[data.billings.findIndex(x => x.id == id)] = newBill; } else { data.billings.push(newBill); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('billingModal'); showToast('Billing saved successfully'); }
});

document.getElementById('accountForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('accId').value;
    const newAcc = { id: id || generateId(), name: document.getElementById('accName').value, email: document.getElementById('accEmail').value, password: document.getElementById('accPassword').value, role: document.getElementById('accRole').value };
    if (id) { data.users[data.users.findIndex(x => x.id == id)] = newAcc; } else { data.users.push(newAcc); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('accountModal'); showToast('Account saved successfully'); }
});

let deleteTarget = null;
function promptDelete(type, id) { if(currentUser.role !== 'admin') return; deleteTarget = { type, id }; document.getElementById('deleteModal').classList.remove('hidden'); }
document.getElementById('confirmDeleteBtn').addEventListener('click', async (e) => {
    if (deleteTarget && currentUser.role === 'admin') {
        const btn = e.target; setBtnLoading(btn, true);
        data[deleteTarget.type] = data[deleteTarget.type].filter(item => item.id != deleteTarget.id);
        const success = await saveData(); setBtnLoading(btn, false);
        if(success) { hideModal('deleteModal'); showToast('Item deleted successfully'); }
    }
});