<?php
// api.php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST");
header("Access-Control-Allow-Headers: Content-Type");

$file = 'data.json';

// --- 1. Helper: Load Data ---
function loadJsonData($file) {
    if (!file_exists($file)) {
        return ["packages" => [], "clients" => [], "billings" => [], "users" => []];
    }
    return json_decode(file_get_contents($file), true);
}

// --- 2. Input Handling ---
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$data = loadJsonData($file);

// --- 3. Handle Login (POST) ---
if ($method === 'POST' && $action === 'login') {
    $input = json_decode(file_get_contents('php://input'), true);
    $email = $input['email'] ?? '';
    $password = $input['password'] ?? '';

    foreach ($data['users'] as $user) {
        if ($user['email'] === $email && $user['password'] === $password) {
            unset($user['password']);
            echo json_encode(["status" => "success", "user" => $user]);
            exit;
        }
    }
    
    http_response_code(401);
    echo json_encode(["status" => "error", "message" => "Invalid credentials"]);
    exit;
}

// --- 4. Handle Data Fetching (GET) ---
if ($method === 'GET' && $action === 'get_data') {
    $userId = $_GET['user_id'] ?? '';
    
    $currentUser = null;
    foreach ($data['users'] as $u) {
        if ($u['id'] === $userId) {
            $currentUser = $u;
            break;
        }
    }

    if (!$currentUser) {
        http_response_code(403);
        echo json_encode(["status" => "error", "message" => "User not found"]);
        exit;
    }

    // A. ADMIN: Sees Everything
    if ($currentUser['role'] === 'admin') {
        echo json_encode($data);
        exit;
    }

    // B. CLIENT: Sees ONLY their data
    if ($currentUser['role'] === 'client') {
        $filteredData = [
            "packages" => $data['packages'], // Public info
            "users" => [$currentUser],       // Only their own user account
            "clients" => [],
            "billings" => []
        ];

        $myClientProfile = null;
        foreach ($data['clients'] as $client) {
            if ($client['name'] === $currentUser['name']) { // Matching by Name linkage
                $filteredData['clients'][] = $client;
                $myClientProfile = $client;
                break; 
            }
        }

        if ($myClientProfile) {
            foreach ($data['billings'] as $bill) {
                if ($bill['clientId'] === $myClientProfile['id']) {
                    $filteredData['billings'][] = $bill;
                }
            }
        }

        echo json_encode($filteredData);
        exit;
    }
}

// --- 5. Handle Save Data (POST) ---
if ($method === 'POST' && $action === 'save_data') {
    $userId = $_GET['user_id'] ?? '';
    
    // Find the user making the request
    $requestingUser = null;
    foreach ($data['users'] as $u) {
        if ($u['id'] === $userId) {
            $requestingUser = $u;
            break;
        }
    }

    if (!$requestingUser) {
        http_response_code(403);
        echo json_encode(["status" => "error", "message" => "User unknown"]);
        exit;
    }

    $newDataJson = file_get_contents('php://input');
    $inputData = json_decode($newDataJson, true);

    if ($inputData === null) {
        http_response_code(400);
        echo json_encode(["status" => "error", "message" => "Invalid JSON"]);
        exit;
    }

    // --- LOGIC A: ADMIN SAVE (Overwrites everything) ---
    if ($requestingUser['role'] === 'admin') {
        if (file_put_contents($file, $newDataJson)) {
            echo json_encode(["status" => "success", "message" => "Data saved"]);
        } else {
            http_response_code(500);
            echo json_encode(["status" => "error", "message" => "Failed to write file"]);
        }
        exit;
    }

    // --- LOGIC B: CLIENT SAVE (Updates ONLY their profile) ---
    if ($requestingUser['role'] === 'client') {
        // We do NOT use the $inputData fully. We only extract the client profile changes.
        // We load the current FULL database ($data) and modify it carefully.

        // 1. Find the client profile sent in the request (assuming they sent their profile in 'clients' array)
        $clientUpdate = null;
        if (isset($inputData['clients'][0])) {
            $clientUpdate = $inputData['clients'][0];
        }

        if (!$clientUpdate) {
            echo json_encode(["status" => "error", "message" => "No profile data sent"]);
            exit;
        }

        // 2. Find the REAL index of this client in the master database
        $foundIndex = -1;
        foreach ($data['clients'] as $index => $c) {
            // Match by name linkage to the logged-in user
            if ($c['name'] === $requestingUser['name']) {
                $foundIndex = $index;
                break;
            }
        }

        if ($foundIndex !== -1) {
            // 3. Update ONLY safe fields. Ignore packageId, status, id changes.
            $data['clients'][$foundIndex]['address'] = $clientUpdate['address'];
            $data['clients'][$foundIndex]['lat'] = $clientUpdate['lat'];
            $data['clients'][$foundIndex]['lng'] = $clientUpdate['lng'];
            // We can allow name update, but it might break linkage if User name isn't also updated. 
            // For now, let's allow Name update in the Client Profile object.
            $data['clients'][$foundIndex]['name'] = $clientUpdate['name'];
            
            // Note: If you want to sync the User Name, you'd need to update $data['users'] too. 
            // But usually, username/login is separate from Display Profile Name.
            
            // 4. Save the modified MASTER database
            if (file_put_contents($file, json_encode($data))) {
                echo json_encode(["status" => "success", "message" => "Profile updated"]);
            } else {
                http_response_code(500);
                echo json_encode(["status" => "error", "message" => "Failed to save profile"]);
            }
        } else {
            echo json_encode(["status" => "error", "message" => "Client profile not found"]);
        }
        exit;
    }
}

echo json_encode(["message" => "API Ready"]);
?>