const API_URL = 'api.php'; 
let data = { packages: [], clients: [], billings: [], users: [] };
let currentUser = null; 

// --- STATE MANAGEMENT for Pagination & Filters ---
let state = {
    clients: {
        page: 1,
        limit: 10
    },
    billings: {
        page: 1,
        limit: 10,
        filterClient: '',
        filterType: '',
        filterMonth: '',
        filterStatus: ''
    }
};

// --- 1. Session Management ---
window.addEventListener('DOMContentLoaded', () => {
    checkSession();
});

function checkSession() {
    const session = localStorage.getItem('sumberejo_session');
    if (session) {
        currentUser = JSON.parse(session);
        loadData().then((success) => {
            if(success) initAppUI();
            else logout();
        });
    } else {
        showLogin();
    }
}

function saveSession(user) {
    currentUser = user;
    localStorage.setItem('sumberejo_session', JSON.stringify(currentUser));
}

function logout() {
    localStorage.removeItem('sumberejo_session');
    currentUser = null;
    showToast('Signed out successfully', 'info');
    setTimeout(() => {
        window.location.reload();
    }, 500);
}

function showLogin() {
    document.getElementById('app-view').classList.add('d-none-custom');
    document.getElementById('login-view').classList.remove('d-none-custom');
}

function initAppUI() {
    document.getElementById('user-display-name').textContent = currentUser.name;
    document.getElementById('user-display-role').textContent = currentUser.role === 'admin' ? 'Administrator' : 'Client';
    document.getElementById('user-avatar').textContent = currentUser.name.charAt(0).toUpperCase();

    document.getElementById('login-view').classList.add('d-none-custom');
    document.getElementById('app-view').classList.remove('d-none-custom');
    
    if (currentUser.role === 'client') {
        document.getElementById('menu-admin').classList.add('d-none-custom');
        document.getElementById('menu-client').classList.remove('d-none-custom');
        document.querySelectorAll('.admin-page').forEach(el => el.classList.add('d-none-custom'));
        renderClientDashboard();
        showPage('client-dashboard');
    } else {
        document.getElementById('menu-admin').classList.remove('d-none-custom');
        document.getElementById('menu-client').classList.add('d-none-custom');
        refreshUI();
    }
}

// --- 2. Client Dashboard Logic ---
function renderClientDashboard() {
    const myClient = data.clients[0]; 
    if (!myClient) {
        showToast("Profile not found", "error");
        return;
    }

    document.getElementById('cpName').value = myClient.name;
    document.getElementById('cpAddress').value = myClient.address;
    document.getElementById('cpLat').value = myClient.lat || '';
    document.getElementById('cpLng').value = myClient.lng || '';
    
    const myPkg = data.packages.find(p => p.id === myClient.packageId);
    if (myPkg) {
        document.getElementById('cpPackageName').textContent = myPkg.name;
        document.getElementById('cpPackageSpeed').textContent = myPkg.speed;
        document.getElementById('cpPackagePrice').textContent = parseInt(myPkg.price).toLocaleString();
    } else {
        document.getElementById('cpPackageName').textContent = "Unknown";
    }
    document.getElementById('cpStatus').textContent = myClient.status;

    const tbody = document.querySelector('#clientBillingTable tbody');
    tbody.innerHTML = '';
    const myBills = data.billings.sort((a,b) => new Date(b.dueDate) - new Date(a.dueDate));
    
    if(myBills.length === 0) {
        tbody.innerHTML = `<tr><td colspan="4" class="p-4 text-center text-gray-400">No billing history found.</td></tr>`;
    } else {
        myBills.forEach(bill => {
            const statusClass = bill.status === 'Paid' ? 'text-green-600 bg-green-50' : 'text-yellow-600 bg-yellow-50';
            const statusIcon = bill.status === 'Paid' ? '<i class="fas fa-check-circle mr-1"></i>' : '<i class="fas fa-clock mr-1"></i>';
            tbody.innerHTML += `
                <tr class="hover:bg-gray-50/50 transition">
                    <td class="p-4 font-medium text-gray-800">${bill.type} ${bill.month !== '-' ? '('+bill.month+')' : ''}</td>
                    <td class="p-4">${bill.dueDate}</td>
                    <td class="p-4 font-bold">Rp ${parseInt(bill.amount).toLocaleString()}</td>
                    <td class="p-4 text-right">
                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${statusClass}">
                            ${statusIcon} ${bill.status}
                        </span>
                    </td>
                </tr>
            `;
        });
    }
}

document.getElementById('clientProfileForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    const btn = e.target.querySelector('button');
    setBtnLoading(btn, true);

    if (data.clients[0]) {
        data.clients[0].name = document.getElementById('cpName').value;
        data.clients[0].address = document.getElementById('cpAddress').value;
        data.clients[0].lat = document.getElementById('cpLat').value;
        data.clients[0].lng = document.getElementById('cpLng').value;
    }

    const success = await saveData();
    setBtnLoading(btn, false);
    if (success) showToast('Profile updated successfully');
});

// --- 3. UX Helpers ---
function showToast(message, type = 'success') {
    const container = document.getElementById('toast-container');
    let bgClass, icon;
    if (type === 'success') {
        bgClass = 'bg-white border-l-4 border-green-500 text-gray-800';
        icon = '<i class="fas fa-check-circle text-green-500 text-xl"></i>';
    } else if (type === 'error') {
        bgClass = 'bg-white border-l-4 border-red-500 text-gray-800';
        icon = '<i class="fas fa-exclamation-circle text-red-500 text-xl"></i>';
    } else {
        bgClass = 'bg-white border-l-4 border-blue-500 text-gray-800';
        icon = '<i class="fas fa-info-circle text-blue-500 text-xl"></i>';
    }

    const toast = document.createElement('div');
    toast.className = `${bgClass} shadow-lg rounded-r-lg p-4 flex items-center gap-3 transform transition-all duration-300 translate-x-full opacity-0 min-w-[300px]`;
    toast.innerHTML = `<div>${icon}</div><div class="font-medium text-sm">${message}</div>`;
    container.appendChild(toast);
    requestAnimationFrame(() => toast.classList.remove('translate-x-full', 'opacity-0'));
    setTimeout(() => {
        toast.classList.add('translate-x-full', 'opacity-0');
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

function setBtnLoading(btn, isLoading) {
    if (isLoading) {
        if (!btn.dataset.originalText) btn.dataset.originalText = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-circle-notch fa-spin mr-2"></i> Processing...';
        btn.disabled = true;
        btn.classList.add('opacity-75', 'cursor-not-allowed');
    } else {
        btn.innerHTML = btn.dataset.originalText;
        btn.disabled = false;
        btn.classList.remove('opacity-75', 'cursor-not-allowed');
    }
}

// --- 4. Login Logic ---
document.getElementById('loginForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    const btn = e.target.querySelector('button');
    const email = document.getElementById('loginEmail').value;
    const pass = document.getElementById('loginPassword').value;

    setBtnLoading(btn, true);
    
    try {
        const response = await fetch(`${API_URL}?action=login`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ email: email, password: pass })
        });

        const resData = await response.json();

        if (response.ok && resData.status === 'success') {
            saveSession(resData.user);
            await loadData();
            initAppUI();
            showToast(`Welcome back, ${resData.user.name}!`);
            setTimeout(() => setBtnLoading(btn, false), 500); 
        } else {
            throw new Error(resData.message || 'Login failed');
        }
    } catch (error) {
        showToast(error.message, 'error');
        setBtnLoading(btn, false);
    }
});

// --- 5. Data Logic ---
async function loadData() {
    if (!currentUser) return false;
    try {
        const response = await fetch(`${API_URL}?action=get_data&user_id=${currentUser.id}`);
        if (!response.ok) throw new Error('Failed to load data');
        const loadedData = await response.json();
        
        data.packages = loadedData.packages || [];
        data.clients = loadedData.clients || [];
        data.billings = loadedData.billings || [];
        data.users = loadedData.users || [];
        return true;
    } catch (error) {
        console.error("Error loading data:", error);
        showToast("Failed to load data from server", "error");
        return false;
    }
}

async function saveData() {
    if(!currentUser) return false;
    try {
        await new Promise(r => setTimeout(r, 600));
        const response = await fetch(`${API_URL}?action=save_data&user_id=${currentUser.id}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });
        if(!response.ok) throw new Error("Save failed");
        
        if (currentUser.role === 'admin') refreshUI();
        return true;
    } catch (error) { 
        console.error("Error saving:", error); 
        showToast("Failed to save changes", "error");
        return false;
    }
}

// --- 6. Admin Render Logic ---
function refreshUI() {
    if (currentUser.role !== 'admin') return; 

    document.getElementById('stat-packages').textContent = data.packages.length;
    document.getElementById('stat-clients').textContent = data.clients.filter(c => c.status === 'Active').length;
    document.getElementById('stat-bills').textContent = data.billings.filter(b => b.status === 'Paid').length;

    renderPackages(true);
    
    // Initial renders with default pagination
    renderClients(true);
    
    // Populate Billing Client Filter Dropdown
    populateSelect('filterBillClient', data.clients);
    renderBillings(true);
    
    renderAccounts(); 
}

function renderPackages(isAdmin) {
    const tbody = document.querySelector('#packagesTable tbody');
    tbody.innerHTML = '';
    data.packages.forEach(pkg => {
        const actions = isAdmin ? 
            `<td class="p-4 text-right">
                <button class="text-blue-500 hover:text-blue-700 mx-1" onclick='openModal("packageModal", ${JSON.stringify(pkg)})'><i class="fas fa-pen"></i></button>
                <button class="text-red-500 hover:text-red-700 mx-1" onclick='promptDelete("packages", "${pkg.id}")'><i class="fas fa-trash"></i></button>
            </td>` : '';
        tbody.innerHTML += `<tr><td class="p-4 font-bold text-gray-800">${pkg.name}</td><td class="p-4"><span class="bg-gray-100 text-gray-800 text-xs px-2 py-1 rounded border border-gray-200">${pkg.speed} Mbps</span></td><td class="p-4 text-gray-500">Rp ${parseInt(pkg.price).toLocaleString()}</td>${actions}</tr>`;
    });
}

// --- CLIENTS RENDER with PAGINATION ---
function updateClientPagination() {
    state.clients.limit = document.getElementById('clientRows').value;
    state.clients.page = 1; // Reset to page 1 on limit change
    renderClients(true);
}

function changeClientPage(direction) {
    const limit = state.clients.limit === 'all' ? data.clients.length : parseInt(state.clients.limit);
    const totalPages = Math.ceil(data.clients.length / limit);
    
    let newPage = state.clients.page + direction;
    if (newPage >= 1 && newPage <= totalPages) {
        state.clients.page = newPage;
        renderClients(true);
    }
}

function renderClients(isAdmin) {
    // 1. STAT CARDS (Unchanged)
    document.getElementById('stat-cli-total').textContent = data.clients.length;
    document.getElementById('stat-cli-active').textContent = data.clients.filter(c => c.status === 'Active').length;
    document.getElementById('stat-cli-inactive').textContent = data.clients.filter(c => c.status === 'Inactive').length;

    // 2. PAGINATION LOGIC
    const limit = state.clients.limit === 'all' ? data.clients.length : parseInt(state.clients.limit);
    const startIndex = (state.clients.page - 1) * limit;
    const endIndex = startIndex + limit;
    const paginatedClients = data.clients.slice(startIndex, endIndex);

    // 3. RENDER TABLE
    const tbody = document.querySelector('#clientsTable tbody');
    tbody.innerHTML = '';
    
    if (paginatedClients.length === 0) {
        tbody.innerHTML = `<tr><td colspan="5" class="p-4 text-center text-gray-400">No clients found.</td></tr>`;
    }

    paginatedClients.forEach(cli => {
        const pkg = data.packages.find(p => p.id == cli.packageId);
        const pkgName = pkg ? pkg.name : '<span class="text-red-500 text-xs">Deleted</span>';
        let badgeClass = cli.status === 'Active' ? 'bg-green-100 text-green-700' : 'bg-gray-200 text-gray-600';
        
        const actions = isAdmin ? 
            `<td class="p-4 text-right">
                <button class="text-yellow-500 hover:text-yellow-600 mx-1" title="Manage Login" onclick='openClientAccountModal(${JSON.stringify(cli)})'><i class="fas fa-user-lock"></i></button>
                <button class="text-blue-500 hover:text-blue-700 mx-1" title="Edit Details" onclick='openModal("clientModal", ${JSON.stringify(cli)})'><i class="fas fa-pen"></i></button>
                <button class="text-red-500 hover:text-red-700 mx-1" title="Delete" onclick='promptDelete("clients", "${cli.id}")'><i class="fas fa-trash"></i></button>
            </td>` : '';
        tbody.innerHTML += `<tr><td class="p-4 font-bold text-gray-800">${cli.name}</td><td class="p-4 text-gray-500 text-sm">${pkgName}</td><td class="p-4 text-gray-500 text-sm">${cli.address}</td><td class="p-4"><span class="${badgeClass} px-3 py-1 rounded-full text-xs font-medium">${cli.status}</span></td>${actions}</tr>`;
    });

    // 4. UPDATE CONTROLS
    const totalItems = data.clients.length;
    document.getElementById('clientPageInfo').textContent = `Showing ${Math.min(startIndex + 1, totalItems)} to ${Math.min(endIndex, totalItems)} of ${totalItems}`;
    
    const totalPages = Math.ceil(totalItems / limit);
    document.getElementById('btnCliPrev').disabled = state.clients.page <= 1;
    document.getElementById('btnCliNext').disabled = state.clients.page >= totalPages;
}

// --- BILLINGS RENDER with FILTERS & PAGINATION ---
function toggleBillingFilterMonth() {
    const type = document.getElementById('filterBillType').value;
    const monthContainer = document.getElementById('filterBillMonthContainer');
    
    if (type === 'Monthly Bill') {
        monthContainer.classList.remove('hidden');
    } else {
        monthContainer.classList.add('hidden');
        document.getElementById('filterBillMonth').value = ""; // Reset month if hidden
    }
    updateBillingPagination(); // Trigger filter update
}

function updateBillingPagination() {
    // Read all filter inputs
    state.billings.limit = document.getElementById('billingRows').value;
    state.billings.filterClient = document.getElementById('filterBillClient').value;
    state.billings.filterType = document.getElementById('filterBillType').value;
    state.billings.filterMonth = document.getElementById('filterBillMonth').value;
    state.billings.filterStatus = document.getElementById('filterBillStatus').value;
    
    state.billings.page = 1; // Reset to page 1 on any filter change
    renderBillings(true);
}

function changeBillingPage(direction) {
    // We need to know totalFilteredItems to check bounds, so we re-run logic effectively inside render
    // Ideally we store filtered list length in state, but recalculating is fast enough here.
    state.billings.page += direction;
    renderBillings(true); 
}

function renderBillings(isAdmin) {
    // 1. STAT CARDS (Unchanged)
    document.getElementById('stat-bill-total').textContent = data.billings.length;
    document.getElementById('stat-bill-paid').textContent = data.billings.filter(b => b.status === 'Paid').length;
    document.getElementById('stat-bill-unpaid').textContent = data.billings.filter(b => b.status === 'Unpaid').length;

    // 2. FILTER LOGIC
    let filteredBills = data.billings.filter(bill => {
        let match = true;
        if (state.billings.filterClient && bill.clientId !== state.billings.filterClient) match = false;
        if (state.billings.filterType && bill.type !== state.billings.filterType) match = false;
        if (state.billings.filterType === 'Monthly Bill' && state.billings.filterMonth && bill.month !== state.billings.filterMonth) match = false;
        if (state.billings.filterStatus && bill.status !== state.billings.filterStatus) match = false;
        return match;
    });

    // 3. PAGINATION LOGIC
    const limit = state.billings.limit === 'all' ? filteredBills.length : parseInt(state.billings.limit);
    const totalItems = filteredBills.length;
    const totalPages = Math.ceil(totalItems / limit) || 1;

    // Boundary Check (Fix for clicking next/prev blindly)
    if (state.billings.page < 1) state.billings.page = 1;
    if (state.billings.page > totalPages) state.billings.page = totalPages;

    const startIndex = (state.billings.page - 1) * limit;
    const endIndex = startIndex + limit;
    const paginatedBills = filteredBills.slice(startIndex, endIndex);

    // 4. RENDER TABLE
    const tbody = document.querySelector('#billingsTable tbody');
    tbody.innerHTML = '';

    if (paginatedBills.length === 0) {
        tbody.innerHTML = `<tr><td colspan="6" class="p-4 text-center text-gray-400">No billings found matching criteria.</td></tr>`;
    }

    paginatedBills.forEach(bill => {
        const cli = data.clients.find(c => c.id == bill.clientId);
        const cliName = cli ? cli.name : 'Unknown';
        const statusBadge = bill.status === 'Paid' ? 'bg-green-100 text-green-700' : 'bg-yellow-100 text-yellow-700';
        let details = `<span class="font-medium text-gray-800">${bill.type}</span>` + (bill.type === 'Monthly Bill' && bill.month ? ` <span class="text-gray-400 text-xs">(${bill.month})</span>` : '');
        const actions = isAdmin ? 
            `<td class="p-4 text-right"><button class="text-blue-500 hover:text-blue-700 mx-1" onclick='openModal("billingModal", ${JSON.stringify(bill)})'><i class="fas fa-pen"></i></button><button class="text-red-500 hover:text-red-700 mx-1" onclick='promptDelete("billings", "${bill.id}")'><i class="fas fa-trash"></i></button></td>` : '';
        tbody.innerHTML += `<tr><td class="p-4 font-bold text-gray-800">${cliName}</td><td class="p-4">${details}</td><td class="p-4 text-gray-500 font-medium">Rp ${parseInt(bill.amount).toLocaleString()}</td><td class="p-4 text-sm text-gray-500">${bill.dueDate}</td><td class="p-4"><span class="${statusBadge} px-3 py-1 rounded-full text-xs font-medium">${bill.status}</span></td>${actions}</tr>`;
    });

    // 5. UPDATE CONTROLS
    // Handle "Showing 0 to 0" if totalItems is 0
    let startDisplay = totalItems === 0 ? 0 : startIndex + 1;
    let endDisplay = Math.min(endIndex, totalItems);
    
    document.getElementById('billingPageInfo').textContent = `Showing ${startDisplay} to ${endDisplay} of ${totalItems}`;
    document.getElementById('btnBillPrev').disabled = state.billings.page <= 1;
    document.getElementById('btnBillNext').disabled = state.billings.page >= totalPages;
}

function renderAccounts() {
    const tbody = document.querySelector('#accountsTable tbody');
    tbody.innerHTML = '';
    const adminUsers = data.users.filter(u => u.role === 'admin');
    
    if (adminUsers.length === 0) {
        tbody.innerHTML = `<tr><td colspan="5" class="p-4 text-center text-gray-400">No administrator accounts found.</td></tr>`;
    }

    adminUsers.forEach(user => {
        const roleBadge = 'bg-blue-100 text-blue-700';
        const actions = `<td class="p-4 text-right">
            <button class="text-blue-500 hover:text-blue-700 mx-1" onclick='openModal("accountModal", ${JSON.stringify(user)})'><i class="fas fa-pen"></i></button>
            <button class="text-red-500 hover:text-red-700 mx-1" onclick='promptDelete("users", "${user.id}")'><i class="fas fa-trash"></i></button>
        </td>`;
        tbody.innerHTML += `
            <tr>
                <td class="p-4 font-bold text-gray-800">${user.name}</td>
                <td class="p-4 text-gray-500">${user.email}</td>
                <td class="p-4"><span class="${roleBadge} px-3 py-1 rounded-full text-xs font-medium uppercase">Admin</span></td>
                <td class="p-4 text-gray-400 text-sm">******</td>
                ${actions}
            </tr>`;
    });
}

// --- Helpers & Modals ---
function toggleSidebar() { 
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('sidebar-overlay');
    if (sidebar.classList.contains('-translate-x-full')) {
        sidebar.classList.remove('-translate-x-full');
        overlay.classList.remove('hidden');
    } else {
        sidebar.classList.add('-translate-x-full');
        overlay.classList.add('hidden');
    }
}

function showPage(pageName) {
    if (currentUser.role === 'client' && pageName !== 'client-dashboard') return;
    document.querySelectorAll('[id^="page-"]').forEach(el => el.classList.add('d-none-custom'));
    document.querySelectorAll('[id^="nav-"]').forEach(el => {
        el.classList.remove('bg-blue-50', 'text-blue-600'); 
        el.classList.add('text-gray-600');
    });

    const targetPage = document.getElementById(`page-${pageName}`);
    if(targetPage) targetPage.classList.remove('d-none-custom');
    
    const activeNav = document.getElementById(`nav-${pageName}`);
    if(activeNav) {
        activeNav.classList.remove('text-gray-600');
        activeNav.classList.add('bg-blue-50', 'text-blue-600');
    }
    if(window.innerWidth < 1024) { 
        document.getElementById('sidebar').classList.add('-translate-x-full'); 
        document.getElementById('sidebar-overlay').classList.add('hidden'); 
    }
}

function generateId() { return Date.now().toString(36) + Math.random().toString(36).substr(2); }
function hideModal(id) { document.getElementById(id).classList.add('hidden'); }
function toggleMonthSelect() { 
    const type = document.getElementById('billType').value;
    const container = document.getElementById('billMonthContainer');
    if(type === 'Monthly Bill') container.classList.remove('hidden');
    else container.classList.add('hidden');
}

function toggleClientStatusDate() {
    const status = document.getElementById('cliStatus').value;
    const container = document.getElementById('cliStatusDateContainer');
    const label = document.getElementById('cliStatusDateLabel');

    if (status === 'Active' || status === 'Inactive' || status === 'Suspend') {
        container.classList.remove('hidden');
        label.textContent = status + ' Date';
    } else {
        container.classList.add('hidden');
    }
}

// NEW: Switch tabs inside client modal
function switchClientTab(tabName) {
    const detailsView = document.getElementById('view-details');
    const historyView = document.getElementById('view-history');
    const tabDetails = document.getElementById('tab-details');
    const tabHistory = document.getElementById('tab-history');

    if (tabName === 'details') {
        detailsView.classList.remove('hidden');
        historyView.classList.add('hidden');
        tabDetails.classList.add('text-blue-600', 'border-b-2', 'border-blue-600');
        tabDetails.classList.remove('text-gray-500');
        tabHistory.classList.remove('text-blue-600', 'border-b-2', 'border-blue-600');
        tabHistory.classList.add('text-gray-500');
    } else {
        detailsView.classList.add('hidden');
        historyView.classList.remove('hidden');
        tabHistory.classList.add('text-blue-600', 'border-b-2', 'border-blue-600');
        tabHistory.classList.remove('text-gray-500');
        tabDetails.classList.remove('text-blue-600', 'border-b-2', 'border-blue-600');
        tabDetails.classList.add('text-gray-500');
    }
}

function populateSelect(elementId, items) { 
    const s = document.getElementById(elementId); 
    // Reuse safe approach: if element doesn't exist (like on initial load before modal open), skip
    if(!s) return;
    
    // Check if there is an initial placeholder option, preserve it if specific logic requires, 
    // but usually we reset. For filters, we have a specific default "All Clients".
    // For Modals, we have "Select...".
    
    if (elementId === 'filterBillClient') {
        s.innerHTML = '<option value="">All Clients</option>';
    } else {
        s.innerHTML = '<option value="">Select...</option>';
    }
    
    items.forEach(i => s.innerHTML += `<option value="${i.id}">${i.name}</option>`); 
}

function openClientAccountModal(clientData) {
    if (currentUser.role !== 'admin') return;
    const existingUser = data.users.find(u => 
        (u.clientId && u.clientId === clientData.id) || 
        (!u.clientId && u.name === clientData.name)
    );
    document.getElementById('caDisplayName').textContent = clientData.name;
    document.getElementById('caClientName').value = clientData.name;
    document.getElementById('caClientId').value = clientData.id; 
    document.getElementById('caEmail').value = existingUser ? existingUser.email : '';
    document.getElementById('caPassword').value = existingUser ? existingUser.password : '';
    document.getElementById('clientAccountModal').classList.remove('hidden');
}

function openModal(id, editData = null) {
    if (currentUser.role !== 'admin') return; 
    
    // Reset any tab state for client modal
    if(id === 'clientModal') switchClientTab('details');

    if(id === 'packageModal') {
        document.getElementById('pkgId').value = editData ? editData.id : '';
        document.getElementById('pkgName').value = editData ? editData.name : '';
        document.getElementById('pkgSpeed').value = editData ? editData.speed : '';
        document.getElementById('pkgPrice').value = editData ? editData.price : '';
    } else if (id === 'clientModal') {
        populateSelect('cliPackage', data.packages);
        document.getElementById('cliId').value = editData ? editData.id : '';
        document.getElementById('cliName').value = editData ? editData.name : '';
        document.getElementById('cliPackage').value = editData ? editData.packageId : '';
        document.getElementById('cliAddress').value = editData ? editData.address : '';
        document.getElementById('cliLat').value = editData ? editData.lat : '';
        document.getElementById('cliLng').value = editData ? editData.lng : '';
        
        // Status Handling
        document.getElementById('cliStatus').value = editData ? editData.status : 'Active';
        document.getElementById('cliStatusDate').value = new Date().toISOString().split('T')[0]; // Default to today for new entry
        toggleClientStatusDate();

        // Populate History Log Table
        const historyTableBody = document.querySelector('#clientHistoryTable tbody');
        const noHistoryMsg = document.getElementById('noHistoryMsg');
        historyTableBody.innerHTML = '';
        
        if (editData && editData.statusHistory && editData.statusHistory.length > 0) {
            noHistoryMsg.classList.add('hidden');
            // Sort history by Date descending
            const sortedHistory = [...editData.statusHistory].sort((a, b) => new Date(b.date) - new Date(a.date));
            sortedHistory.forEach(h => {
                let badgeColor = h.status === 'Active' ? 'text-green-600' : 'text-gray-500';
                historyTableBody.innerHTML += `
                    <tr>
                        <td class="py-2 font-medium ${badgeColor}">${h.status}</td>
                        <td class="py-2">${h.date}</td>
                        <td class="py-2 text-right text-xs text-gray-400">${new Date(h.timestamp).toLocaleDateString()}</td>
                    </tr>
                `;
            });
        } else {
            noHistoryMsg.classList.remove('hidden');
        }

    } else if (id === 'billingModal') {
        populateSelect('billClient', data.clients);
        document.getElementById('billId').value = editData ? editData.id : '';
        document.getElementById('billClient').value = editData ? editData.clientId : '';
        document.getElementById('billType').value = editData ? editData.type : 'Monthly Bill';
        document.getElementById('billAmount').value = editData ? editData.amount : '';
        document.getElementById('billMonth').value = editData ? editData.month : 'January';
        document.getElementById('billDate').value = editData ? editData.dueDate : '';
        document.getElementById('billStatus').value = editData ? editData.status : 'Unpaid';
        toggleMonthSelect();
    } else if (id === 'accountModal') {
        document.getElementById('accId').value = editData ? editData.id : '';
        document.getElementById('accName').value = editData ? editData.name : '';
        document.getElementById('accEmail').value = editData ? editData.email : '';
        document.getElementById('accPassword').value = editData ? editData.password : '';
    }
    document.getElementById(id).classList.remove('hidden');
}

// --- Submit Handlers ---

document.getElementById('clientAccountForm').addEventListener('submit', async (e) => {
    e.preventDefault(); 
    if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); 
    setBtnLoading(btn, true);
    
    const clientName = document.getElementById('caClientName').value;
    const clientId = document.getElementById('caClientId').value; 
    const email = document.getElementById('caEmail').value;
    const password = document.getElementById('caPassword').value;
    
    const userIndex = data.users.findIndex(u => (u.clientId && u.clientId === clientId) || (!u.clientId && u.name === clientName));

    if (userIndex >= 0) { 
        data.users[userIndex].email = email; 
        data.users[userIndex].password = password; 
        data.users[userIndex].clientId = clientId; 
        data.users[userIndex].name = clientName; 
    } else { 
        data.users.push({ id: generateId(), clientId: clientId, name: clientName, email: email, password: password, role: 'client' }); 
    }

    const success = await saveData(); 
    setBtnLoading(btn, false);
    if(success) { hideModal('clientAccountModal'); showToast('Login credentials updated successfully'); }
});

document.getElementById('accountForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('accId').value;
    const newAcc = { id: id || generateId(), name: document.getElementById('accName').value, email: document.getElementById('accEmail').value, password: document.getElementById('accPassword').value, role: 'admin' };
    if (id) { data.users[data.users.findIndex(x => x.id == id)] = newAcc; } else { data.users.push(newAcc); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('accountModal'); showToast('Admin account saved'); }
});

document.getElementById('packageForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('pkgId').value;
    const newPkg = { id: id || generateId(), name: document.getElementById('pkgName').value, speed: document.getElementById('pkgSpeed').value, price: document.getElementById('pkgPrice').value };
    if (id) { data.packages[data.packages.findIndex(x => x.id == id)] = newPkg; } else { data.packages.push(newPkg); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('packageModal'); showToast('Package saved successfully'); }
});

// UPDATED CLIENT SAVE LOGIC
document.getElementById('clientForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('cliId').value;
    
    // Get Basic Data
    let existingClient = id ? data.clients.find(x => x.id === id) : null;
    let history = existingClient && existingClient.statusHistory ? existingClient.statusHistory : [];

    // Get Status Data
    const currentStatus = document.getElementById('cliStatus').value;
    const currentDateStr = document.getElementById('cliStatusDate').value;

    // Logic: Append to history if status changed or date provided
    // We simply treat every save with a date as a "confirmation" of status at that date
    if (currentStatus && currentDateStr) {
        // Optional: Avoid exact duplicates (same status AND same date)
        const lastEntry = history.length > 0 ? history[history.length - 1] : null;
        const isDuplicate = lastEntry && lastEntry.status === currentStatus && lastEntry.date === currentDateStr;
        
        if (!isDuplicate) {
            history.push({
                status: currentStatus,
                date: currentDateStr,
                timestamp: Date.now() // Record when we physically saved this
            });
        }
    }

    const newCli = { 
        id: id || generateId(), 
        name: document.getElementById('cliName').value, 
        packageId: document.getElementById('cliPackage').value, 
        address: document.getElementById('cliAddress').value, 
        lat: document.getElementById('cliLat').value, 
        lng: document.getElementById('cliLng').value, 
        status: currentStatus,
        statusHistory: history // Save the full history array
    };
    
    if (id) { data.clients[data.clients.findIndex(x => x.id == id)] = newCli; } else { data.clients.push(newCli); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('clientModal'); showToast('Client saved successfully'); }
});

document.getElementById('billingForm').addEventListener('submit', async (e) => {
    e.preventDefault(); if(currentUser.role !== 'admin') return;
    const btn = e.target.querySelector('button'); setBtnLoading(btn, true);
    const id = document.getElementById('billId').value;
    const type = document.getElementById('billType').value;
    const newBill = { id: id || generateId(), clientId: document.getElementById('billClient').value, type: type, amount: document.getElementById('billAmount').value, month: (type === 'Monthly Bill') ? document.getElementById('billMonth').value : '-', dueDate: document.getElementById('billDate').value, status: document.getElementById('billStatus').value };
    if (id) { data.billings[data.billings.findIndex(x => x.id == id)] = newBill; } else { data.billings.push(newBill); }
    const success = await saveData(); setBtnLoading(btn, false);
    if(success) { hideModal('billingModal'); showToast('Billing saved successfully'); }
});

let deleteTarget = null;
function promptDelete(type, id) { if(currentUser.role !== 'admin') return; deleteTarget = { type, id }; document.getElementById('deleteModal').classList.remove('hidden'); }
document.getElementById('confirmDeleteBtn').addEventListener('click', async (e) => {
    if (deleteTarget && currentUser.role === 'admin') {
        const btn = e.target; setBtnLoading(btn, true);
        data[deleteTarget.type] = data[deleteTarget.type].filter(item => item.id != deleteTarget.id);
        const success = await saveData(); setBtnLoading(btn, false);
        if(success) { hideModal('deleteModal'); showToast('Item deleted successfully'); }
    }
});